%% Visualize the simulated agent

TrainingHistory = env.TrainingHistory;

thismap = TurinMaps{10,12}; 
thisdensity = DensityMaps{10,12};
Map = TurinMaps{10,12};


figure;
axis([1,200, 1,200, 0,20]);
hold on


dronePlot = plot3(0, 0, 0, 'ro', 'MarkerSize', 10, 'MarkerFaceColor', 'r');
goalPlot = plot3(0, 0, 0, 'go', 'MarkerSize', 10, 'MarkerFaceColor', 'g');


isoSurfaceObj = isosurface(thismap, 0.5);                                 
patch(isoSurfaceObj, 'FaceColor', 'blue', 'EdgeColor', 'k');            

imagesc(thisdensity);                                                     
colormap(parula);
colorbar;



for i = 1:size(TrainingHistory, 2)

    currentState = TrainingHistory(1, i);                               
    currentState = cell2mat(currentState);

    set(dronePlot, 'XData', currentState(1), 'YData', currentState(2), 'ZData', 20); % upddate the new position of drone
    set(goalPlot, 'XData', currentState(3), 'YData', currentState(4), 'ZData', 20);

    drawnow;
    pause(0.042);

end




%% Evaluating Path Length and Path Noise for RL


TrainingHistory = env.TrainingHistory;


Last2Positions = zeros(2,2);
Path_Length_RL = [];                                                      % to store the length of path (length of each segment of path between each timestep)
waypoint_RL_Noise = [];                                                   % to store the overal noise at each drone point (each timestep)
waypoint_RL = [];                                                         % to store the drone position at each timestep



for i = 1:size(TrainingHistory, 2)

    current_state = TrainingHistory(1, i);                                 
    current_state = current_state{1,1};

    Last2Positions = [current_state(1), current_state(2) ; Last2Positions(1,:)]; 

    waypoint_RL_Noise = [waypoint_RL_Noise; current_state(7)];
    waypoint_RL = [waypoint_RL; current_state(1), current_state(2)];

    if i>1
        distance_travelled = sqrt((Last2Positions(1,1) - Last2Positions(2,1))^2 + (Last2Positions(1,2) - Last2Positions(2,2))^2 );
        Path_Length_RL = [Path_Length_RL; distance_travelled];
    end
end



mean_Path_Length_RL = mean(Path_Length_RL);                               
Total_Path_Length_RL = sum(Path_Length_RL(:));



%% Interpolate points along the flight path


dists = sqrt(sum(diff(waypoint_RL).^2, 2));
cum_dists = [0; cumsum(dists)];

new_distances = 0:0.5:cum_dists(end);

new_points_x = interp1(cum_dists, waypoint_RL(:,1), new_distances, 'linear');
new_points_y = interp1(cum_dists, waypoint_RL(:,2), new_distances, 'linear');

pointsRL = [new_points_x', new_points_y'];


%% assign the corrsponding velocity for all points including interpolated ones


total_distance = cumsum(dists); 

dists_all_points = sqrt(sum(diff(pointsRL).^2, 2));

cum_dists_all_points = [0; cumsum(dists_all_points)];

new_path_Length = zeros(size(pointsRL, 1), 1);


velocity_index = 1; 

for i = 1:length(cum_dists_all_points)

    while velocity_index < length(total_distance) && cum_dists_all_points(i) > total_distance(velocity_index)
        velocity_index = velocity_index + 1; 
    end

    new_path_Length(i) = Path_Length_RL(velocity_index);
end


%% Evaluating Noise values of all points on RL path


Rays_Paths_RL = cell(1, size(pointsRL, 1));                        % initialize cells to store the coordinate and noise value along the ray paths and the overal noise at each timestep of RL for interpolated points
SPLs_RL = cell(1, size(pointsRL, 1));
Noises_RL = zeros(1, size(pointsRL, 1));

for i = 1:size(pointsRL, 1)
    [a, b, c] = RayTracing(pointsRL(i, 1), pointsRL(i, 2), 20, new_path_Length(i), Map);

    Rays_Paths_RL{i} = a;
    SPLs_RL{i} = b;
    Noises_RL(i) = c;

end


all_RL_Noise = Noises_RL';
all_RL_points = pointsRL;


%% Calculating Noise heatmap matrix for RL agent


Noise_heatMap1 = zeros(200,200,20);                                       


for i = 1:size(all_RL_points, 1)                               

    Ray_Path = Rays_Paths_RL(i);                                
    Ray_Path = Ray_Path{1,1};

    SPL = SPLs_RL(i);                             
    SPL = SPL{1,1};

    for j = 1:46

        ray_coordinates = Ray_Path{j};
        ray_noise_values = SPL{j}(:, 1);

        if size(ray_coordinates,1) ~= size(ray_noise_values,1)
            continue
        end


        indices2 = ceil(ray_coordinates);

        for m = 1:size(indices2, 1) 
            Noise_heatMap1(indices2(m, 1), indices2(m, 2), indices2(m, 3)) = Noise_heatMap1(indices2(m, 1), indices2(m, 2), indices2(m, 3)) + 10^(ray_noise_values(m, 1) / 10);
        end
    end
end


Noise_heatMap1 = 10 * log10(Noise_heatMap1);


Noise_Impact1 = zeros(200,200);                                          

for i=1:200
    for j=1:200
        for k=1:20
            Noise_Impact1(i,j) = Noise_Impact1(i,j) + 10^(Noise_heatMap1(i,j,k) / 10); 
        end
    end
end

        
Noise_Impact1 = 10 * log10(Noise_Impact1);



%% Plotting Noise Heat Map for RL agent (overal noise)


max_Noise_RL = max(all_RL_Noise);
min_Noise_RL = min(all_RL_Noise);

x_grid = linspace(1,200, 400); 
y_grid = linspace(1,200, 400); 

[X, Y] = meshgrid(x_grid, y_grid);

F = scatteredInterpolant(all_RL_points(:,1), all_RL_points(:,2), all_RL_Noise, 'linear', 'none');
Z = F(X, Y);

Z_flipped = flipud(Z);

% Create heatmap
figure;
heatmap1 = heatmap(x_grid, y_grid, Z_flipped);


xlabel('X [m]');
ylabel('Y [m]');
title('Heat Map of SPL along the trajectory (RL)')
clim(heatmap1, [36, 58]);
colorbar;
colormap(jet);
grid off;

% Customize X-axis ticks
xTicks = heatmap1.XDisplayLabels;                                         % Get current X-axis tick labels
xTicks(1:400) = {''};                                                     % Remove  the ticks by setting them to empty strings:
xTicks(end) = {'400'};

% Set the tick labels at intervals of 20 in the middle
for i = 1:40:400
    xTicks{i} = num2str(i - 1);                                           % Set tick labels at intervals of 40
end

heatmap1.XDisplayLabels = xTicks;


% Customize Y-axis ticks
yTicks = heatmap1.YDisplayLabels;                                         % Get current Y-axis tick labels
yTicks(1:400) = {''};                                                     % Remove  the ticks by setting them to empty strings:
yTicks(end) = {'400'};

% Set the tick labels at intervals of 20 in the middle
for i = 1:40:400
    yTicks{i} = num2str(i - 1);                                           % Set tick labels at intervals of 40
end

heatmap1.YDisplayLabels = flip(yTicks);





% Contour plot of total noise impact
figure
a = Noise_Impact1;
a(a == 0) = NaN;

contourf(a', 'LineWidth', 0.1, 'EdgeAlpha', 0.5);                         %transposed (a') to get correct xy axis orientation (happens when map's length and height is equal)
xlabel('X [m]');
ylabel('Y [m]');
title('Noise footprint at 2D visualization (RL)');
colormap(jet);
clim( [30, 110]);

cb = colorbar; 
cb.Label.String = 'SPL [dB]';

xTicks = xticklabels;
xTicks = arrayfun(@(x) num2str(str2double(x) * 2), xTicks, 'UniformOutput', false);% Convert the tick labels to numbers, multiply by 2, and convert back to strings
xticklabels(xTicks); 

yTicks = yticklabels; 
yTicks = arrayfun(@(x) num2str(str2double(x) * 2), yTicks, 'UniformOutput', false);% Convert the tick labels to numbers, multiply by 2, and convert back to strings
yticklabels(yTicks); 





%% Evaluating Path Length of A*

                             
waypoint_Astar = optimal_path;                                            % waypoints of path taken by A*

waypoint_Astar = flipud(waypoint_Astar);                                 
waypoint_Astar = waypoint_Astar(:, [2, 1]);                               


Last2Positions = zeros(2,2);
Path_Length_Astar = [];                                                   


for i = 1:size(waypoint_Astar, 1)

    Last2Positions = [waypoint_Astar(i,:) ; Last2Positions(1,:)]; 

    if i>1
        distance_travelled = sqrt((Last2Positions(1,1) - Last2Positions(2,1))^2 + (Last2Positions(1,2) - Last2Positions(2,2))^2 );
        Path_Length_Astar = [Path_Length_Astar; distance_travelled];
    end
end



Total_Path_Length_Astar = sum(Path_Length_Astar(:));



%% Interpolate points along the A* path 


dists = sqrt(sum(diff(waypoint_Astar).^2, 2));

cum_dists = [0; cumsum(dists)];

new_distances = 0:0.5:cum_dists(end);

new_points_x = interp1(cum_dists, waypoint_Astar(:,1), new_distances, 'linear');
new_points_y = interp1(cum_dists, waypoint_Astar(:,2), new_distances, 'linear');

all_Astar_points = [new_points_x', new_points_y'];



%% Evaluating Noise values of A* path


third_column = ones(size(all_Astar_points, 1), 1) * 20;                 
all_Astar_points = [all_Astar_points, third_column];                    


Rays_Paths2 = cell(1, size(all_Astar_points, 1));                        
SPLs2 = cell(1, size(all_Astar_points, 1));
Noises2 = zeros(1, size(all_Astar_points, 1));



for i = 1:size(all_Astar_points, 1)                                      
    [a,b,c] = RayTracing(all_Astar_points(i, 1), all_Astar_points(i, 2), all_Astar_points(i, 3), mean_Path_Length_RL, Map);

    Rays_Paths2{i} = a;
    SPLs2{i} = b;
    Noises2(i) = c;
end

all_Astar_Noise = Noises2';                                            


%% Plotting Noise Impact in 3D for A* (noise at each point of the rays in 3D space)


Noise_heatMap2 = zeros(200,200,20);

Last2Positions = zeros(2,3);                                          



for i = 1:size(all_Astar_points, 1)                                     

    Ray_Path = Rays_Paths2{i};
    SPL = SPLs2{i};       

    Last2Positions = [all_Astar_points(i,:) ; Last2Positions(1,:)];


    for j = 1:46
        ray_coordinates = Ray_Path{j};
        ray_noise_values = SPL{j}(:, 1);

        if size(ray_coordinates,1) ~= size(ray_noise_values,1)          
            continue
        end

        indices2 = ceil(ray_coordinates);

        for m = 1:size(indices2, 1)
            Noise_heatMap2(indices2(m, 1), indices2(m, 2), indices2(m, 3)) = Noise_heatMap2(indices2(m, 1), indices2(m, 2), indices2(m, 3)) + 10^(ray_noise_values(m, 1) / 10);
        end
    end
end


Noise_heatMap2 = 10 * log10(Noise_heatMap2);


Noise_Impact2 = zeros(200,200);                                         

for i=1:200
    for j=1:200
        for k=1:20
            Noise_Impact2(i,j) = Noise_Impact2(i,j) + 10^(Noise_heatMap2(i,j,k) / 10); 
        end
    end
end

        
Noise_Impact2 = 10 * log10(Noise_Impact2);



%% Plotting Noise Heat Map for A* (overal noise)


max_Noise_Astar = max(all_Astar_Noise);
min_Noise_Astar = min(all_Astar_Noise);


x_grid = linspace(1,200, 400); 
y_grid = linspace(1,200, 400); 

[X, Y] = meshgrid(x_grid, y_grid);

F = scatteredInterpolant(all_Astar_points(:,1), all_Astar_points(:,2), all_Astar_Noise, 'linear', 'none');
Z = F(X, Y);

Z_flipped = flipud(Z);

% Create heatmap
figure;
heatmap2 = heatmap(x_grid, y_grid, Z_flipped);


xlabel('X [m]');
ylabel('Y [m]');
title('Heat Map of SPL along the trajectory (A*)')
clim(heatmap2, [44, 56]);
colorbar;
colormap(jet);
grid off;

% Customize X-axis ticks
xTicks = heatmap2.XDisplayLabels; % Get current X-axis tick labels
xTicks(1:400) = {''}; % Remove  the ticks by setting them to empty strings:
xTicks(end) = {'400'};

% Set the tick labels at intervals of 20 in the middle
for i = 1:40:400
    xTicks{i} = num2str(i - 1); % Set tick labels at intervals of 20
end

heatmap2.XDisplayLabels = xTicks;


% Customize Y-axis ticks
yTicks = heatmap2.YDisplayLabels; % Get current X-axis tick labels
yTicks(1:400) = {''}; % Remove  the ticks by setting them to empty strings:
yTicks(end) = {'400'};

% Set the tick labels at intervals of 20 in the middle
for i = 1:40:400
    yTicks{i} = num2str(i - 1); % Set tick labels at intervals of 20
end

heatmap2.YDisplayLabels = flip(yTicks);


% Contour plot of total noise impact
figure
a = Noise_Impact2;
a(a == 0) = NaN;

contourf(a', 'LineWidth', 0.1, 'EdgeAlpha', 0.5);                         % transposed to get correct xy axis orientation (happens when map's length and height is equal)
xlabel('X [m]');
ylabel('Y [m]');
title('Noise footprint at 2D visualization (A*)');
colormap(jet);
clim( [30, 110]);

cb = colorbar; % Create a colorbar
cb.Label.String = 'SPL [dB]';

xTicks = xticklabels; % Get the tick labels of the x-axis
xTicks = arrayfun(@(x) num2str(str2double(x) * 2), xTicks, 'UniformOutput', false);
xticklabels(xTicks); % Set the updated tick labels of the x-axis

yTicks = yticklabels; % Get the tick labels of the x-axis
yTicks = arrayfun(@(x) num2str(str2double(x) * 2), yTicks, 'UniformOutput', false);
yticklabels(yTicks); % Set the updated tick labels of the x-axis



%% Direct Path calculations

start_point = [1,1];                                                     % enter the start and target coordinaets
end_point = [170,170];

waypoint_Direct = [start_point; end_point];

Total_Path_Length_Direct = norm(start_point - end_point);
total_points_desired = Total_Path_Length_Direct * 2;

x_interpolated = interp1([0, 1], [start_point(1), end_point(1)], linspace(0, 1, total_points_desired), 'linear');
y_interpolated = interp1([0, 1], [start_point(2), end_point(2)], linspace(0, 1, total_points_desired), 'linear');


all_Direct_points = [x_interpolated', y_interpolated'];


Rays_Paths_Direct = cell(1, size(all_Direct_points, 1));  % initialize cells to store the coordinate and noise value along the ray paths and the overal noise at each timestep of RL for interpolated points
SPLs_Direct = cell(1, size(all_Direct_points, 1));
Noises_Direct = zeros(1, size(all_Direct_points, 1));


for i = 1:size(all_Direct_points, 1)

    [a, b, c] = RayTracing(all_Direct_points(i, 1), all_Direct_points(i, 2), 20, mean_Path_Length_RL, Map);

    Rays_Paths_Direct{i} = a;
    SPLs_Direct{i} = b;
    Noises_Direct(i) = c;

end


all_Direct_Noise = Noises_Direct';                                 



%% Plotting Noise Impact in 3D for Direct Path (noise at each point of the rays in 3D space)


Noise_heatMap3 = zeros(200,200,20);

Last2Positions = zeros(2,2);                                            



for i = 1:size(all_Direct_points, 1)                                     

    Ray_Path = Rays_Paths_Direct{i};
    SPL = SPLs_Direct{i};        

    Last2Positions = [all_Direct_points(i,:) ; Last2Positions(1,:)];


    for j = 1:46

        ray_coordinates = Ray_Path{j};
        ray_noise_values = SPL{j}(:, 1);

        if size(ray_coordinates,1) ~= size(ray_noise_values,1)         
            continue
        end


        indices2 = ceil(ray_coordinates);

        for m = 1:size(indices2, 1) 
            Noise_heatMap3(indices2(m, 1), indices2(m, 2), indices2(m, 3)) = Noise_heatMap3(indices2(m, 1), indices2(m, 2), indices2(m, 3)) + 10^(ray_noise_values(m, 1) / 10);
        end
    end
end



Noise_heatMap3 = 10 * log10(Noise_heatMap3);


Noise_Impact3 = zeros(200,200);                                       

for i=1:200
    for j=1:200
        for k=1:20
            Noise_Impact3(i,j) = Noise_Impact3(i,j) + 10^(Noise_heatMap3(i,j,k) / 10); 
        end
    end
end

        
Noise_Impact3 = 10 * log10(Noise_Impact3);



max_Noise_Direct = max(all_Direct_Noise);
min_Noise_Direct = min(all_Direct_Noise);


% Contour plot of total noise impact
figure
a = Noise_Impact3;
a(a == 0) = NaN;

contourf(a', 'LineWidth', 0.1, 'EdgeAlpha', 0.5); %transposed to get correct xy axis orientation (happens when map's length and height is equal)
xlabel('X [m]');
ylabel('Y [m]');
title('Noise footprint at 2D visualization (Direct Path)');
colormap(jet);
clim( [30, 110]);

cb = colorbar; 
cb.Label.String = 'SPL [dB]';

xTicks = xticklabels; % Get the tick labels of the x-axis
xTicks = arrayfun(@(x) num2str(str2double(x) * 2), xTicks, 'UniformOutput', false);
xticklabels(xTicks); % Set the updated tick labels of the x-axis

yTicks = yticklabels; % Get the tick labels of the x-axis
yTicks = arrayfun(@(x) num2str(str2double(x) * 2), yTicks, 'UniformOutput', false);
yticklabels(yTicks); % Set the updated tick labels of the x-axis




%% Plotting the trajectories


obstaclesSlice = Map(:,:,1);

[obstacleY, obstacleX] = find(obstaclesSlice);

figure;
[X, Y] = meshgrid(1:200, 1:200);
surf(X, Y, zeros(size(thisdensity)), thisdensity, 'EdgeColor', 'none');
view(2);
colormap('parula');
hold on;

scatter(obstacleX, obstacleY, 20, 'k', 'filled'); 

plot(all_RL_points(:,1), all_RL_points(:,2), 'r-', 'LineWidth', 1.5);
plot(all_Astar_points(:,1), all_Astar_points(:,2), 'y-', 'LineWidth', 1.5);
plot(all_Direct_points(:,1), all_Direct_points(:,2), 'c-', 'LineWidth', 1.5);


plot(start_point(1), start_point(2), 'ro', 'MarkerSize', 8, 'MarkerFaceColor', 'r'); 
plot(end_point(1), end_point(2), 'go', 'MarkerSize', 8, 'MarkerFaceColor', 'g'); 

title('Flight Paths');
xlabel('X [m]');
ylabel('Y [m]');
legend('','','RL', 'A*', 'Direct Path','Start', 'Target');
axis([1,200,1,200]);
cb = colorbar;
cb.Label.String = 'Population Density';

xTicks = xticklabels; % Get the tick labels of the x-axis
xTicks = arrayfun(@(x) num2str(str2double(x) * 2), xTicks, 'UniformOutput', false);
xticklabels(xTicks); % Set the updated tick labels of the x-axis

yTicks = yticklabels; % Get the tick labels of the x-axis
yTicks = arrayfun(@(x) num2str(str2double(x) * 2), yTicks, 'UniformOutput', false);
yticklabels(yTicks); % Set the updated tick labels of the x-axis




%% Evaluating noise per unit length for the trajectories


distances = sqrt(sum(diff(all_RL_points).^2, 2));                        

cumulative_distances = [0; cumsum(distances)];                    

cumulative_noise_RL = trapz(cumulative_distances, all_RL_Noise);

cumulative_noise_RL_per_meter = cumulative_noise_RL / cumulative_distances(end);


distances2 = sqrt(sum(diff(all_Astar_points).^2, 2)); 

cumulative_distances2 = [0; cumsum(distances2)];                        

cumulative_noise_Astar = trapz(cumulative_distances2, all_Astar_Noise);

cumulative_noise_Astar_per_meter = cumulative_noise_Astar /  cumulative_distances2(end);



distances3 = sqrt(sum(diff(all_Direct_points).^2, 2)); 

cumulative_distances3 = [0; cumsum(distances3)];                          

cumulative_noise_Direct = trapz(cumulative_distances3, all_Direct_Noise);

cumulative_noise_Direct_per_meter = cumulative_noise_Direct /  cumulative_distances3(end);


%% Plotting Noise values along the paths


figure;
hold on;

area(cumulative_distances, all_RL_Noise, 'FaceColor', 'b', 'EdgeColor', 'none', 'FaceAlpha', 0.5); 
plot(cumulative_distances, all_RL_Noise, 'k-', 'LineWidth', 1.5); 
hold off

xlabel('Path Length [m]');
ylabel('SPL [dB]');
title('SPL distribution along the Trajectory (RL)');

xTicks = xticklabels; % Get the tick labels of the x-axis
xTicks = arrayfun(@(x) num2str(str2double(x) * 2), xTicks, 'UniformOutput', false);
xticklabels(xTicks); % Set the updated tick labels of the x-axis


figure;
hold on;

area(cumulative_distances2, all_Astar_Noise, 'FaceColor', 'b', 'EdgeColor', 'none', 'FaceAlpha', 0.5); 
plot(cumulative_distances2, all_Astar_Noise, 'k-', 'LineWidth', 1.5); 

xlabel('Path Length [m]');
ylabel('SPL [dB]');
title('SPL distribution along the Trajectory (A*)');

xTicks = xticklabels; % Get the tick labels of the x-axis
xTicks = arrayfun(@(x) num2str(str2double(x) * 2), xTicks, 'UniformOutput', false);
xticklabels(xTicks); % Set the updated tick labels of the x-axis



figure;
hold on;

area(cumulative_distances3, all_Direct_Noise, 'FaceColor', 'b', 'EdgeColor', 'none', 'FaceAlpha', 0.5);
plot(cumulative_distances3, all_Direct_Noise, 'k-', 'LineWidth', 1.5); 

xlabel('Path Length [m]');
ylabel('SPL [dB]');
title('SPL distribution along the Trajectory (Direct Path)');

xTicks = xticklabels; % Get the tick labels of the x-axis
xTicks = arrayfun(@(x) num2str(str2double(x) * 2), xTicks, 'UniformOutput', false);
xticklabels(xTicks); % Set the updated tick labels of the x-axis



%% Ray tracing functions



function [impactCoord, SPL, Noise] = RayTracing(DroneX, DroneY, DroneZ, meanPathLength, Map)             % Noise Ray tracing function

aa = load('RayTracingRays.mat');
rays = aa.r;                                 

velocity = meanPathLength / 0.3 * 3.6 * 2;   
NoiseSource = 60.24 * exp(0.003379 * velocity);            

mapSize = [200,200,20];

N = 46; 
maxDistance = 200;                                          
absoptionCoeffcient = 0.00076 * 2;                         % atmospheric absorbtion (alpha) 0.0011 dB/meter (since each cell is 2 meter and each step is 1 unit, multiply by 2)
stepLength = 0.3;                                          % length of each step between each point while ray tracing (adjust as needed)


RayDistance = cell(N, 1);                                  % Store distances of each segment of each ray  while travelling
impactCoord = cell(N, 1);                                  % Store ccoordinates of each segment of each ray  while travelling
distLoss = cell(N, 1);                                     % sound pressure level loss at each segment of ray path
SPL = cell(N, 1);                                          % Sound pressure level at each point of the rays
ReceiverCells = zeros(N, 1);                               % Noise received at ground from each ray

for iRay = 1:N                                             

    currentX = DroneX;
    currentY = DroneY;                            
    currentZ = DroneZ;

    xDir = rays(iRay,1);
    yDir = rays(iRay,2);                                   % Set one of the ray directions
    zDir = rays(iRay,3);

    ray_distance = 0;                                      % ray travel distance and reflection count. Ray tracing is terminated when the travel distance or reflection count exceeds the threshold.
    counter = 0;                                           % to keep track of which point on the ray is being evaluated in the while loop

    RayDistance{iRay} = [];                                % Initialize the array to store distances, coordinates, distance loss, reflection loss and the SPL value at each segment of this ray
    impactCoord{iRay} = [];
    distLoss{iRay} = [];
    SPL{iRay} = [];                                        % Initialize the array to store distances, coordinates, distance loss, reflection loss and the SPL value at each segment of this ray

    while ray_distance <= maxDistance % for each point there is 3 scenario, it reaches the map boundary or it hits an obstacle or it travelles freely. based on the situation for the point, the corresponding loop is executed.

        currentX = currentX + xDir * stepLength;
        currentY = currentY + yDir * stepLength;
        currentZ = currentZ + zDir * stepLength;

        currentX = max(1, min(currentX, mapSize(1)));    % make sure indices are within range
        currentY = max(1, min(currentY, mapSize(2)));
        currentZ = max(1, min(currentZ, mapSize(3)));

        counter = counter + 1;

        if currentX == 1 || currentX == mapSize(1) || currentY == 1 || currentY == mapSize(2) ||  currentZ == mapSize(3)  

            impactCoord{iRay}(counter,:) = [currentX, currentY, currentZ];   % add the new point coordinate to the list

            if size(impactCoord{iRay}, 1) == 1          
                distance = sqrt((currentX -  DroneX)^2 + (currentY -  DroneY)^2 + (currentZ -  DroneZ)^2);
            else
                distance = sqrt((currentX -  impactCoord{iRay}(end-1,1))^2 + (currentY -  impactCoord{iRay}(end-1,2))^2 + (currentZ -  impactCoord{iRay}(end-1,3))^2);
            end

            RayDistance{iRay}(counter) =  distance;        % store the distance travelled by ray in this segment

            attenuation =  distance * absoptionCoeffcient;   % loss due to atmospheric absorption


            if sum(RayDistance{iRay}) >= 1

                inverseLaw = 20 * log10(sum(RayDistance{iRay}) * 2) + 11;


            else          
                inverseLaw = 0;
            end

            distLoss{iRay}(counter) = attenuation;

            splValue = NoiseSource - sum(distLoss{iRay}) - inverseLaw;

            if splValue <= 0                               % stop the tracing when spl reaches zero
                break
            end

            SPL{iRay}(counter,1) = splValue;               % Calculate SPL at each point

            ray_distance = ray_distance + distance;        % Update ray travel distance

            if round(currentZ) <= 2 && ReceiverCells(iRay) == 0   
                ReceiverCells(iRay,1) = SPL{iRay}(counter,1);
            end

            break;
        end



        if Map(ceil(currentY), ceil(currentX), ceil(currentZ)) == 1 || Map(floor(currentY), floor(currentX), floor(currentZ)) == 1 || currentZ == 1 || ...    % reflection for hitting obstacles (since in map only integer values are set to 1, we check if at the current point, the above and below points are occupied , if so we know we are in occupied point)
                (Map(ceil(currentY), ceil(currentX), min(ceil(currentZ) + 1, size(Map, 3))) == 1 && Map(ceil(currentY), ceil(currentX), max(ceil(currentZ) - 1, 1)) == 1) || ...
                (Map(floor(currentY), floor(currentX), min(floor(currentZ) + 1, size(Map, 3))) == 1 && Map(floor(currentY), floor(currentX), max(floor(currentZ) - 1, 1)) == 1)

            [hit, hitFace] = checkHitFace(currentX, currentY, currentZ);  % get the which face was hit

            if hit                                         % Reflect the ray based on the hit face
                normalVectors = [
                    1, 0, 0;                            
                    -1, 0, 0;                           
                    0, 1, 0;                             
                    0,-1, 0;                            
                    0, 0, 1;                              
                    0, 0,-1                             
                    ];

                hitNormal = normalVectors(hitFace, :);        % normal vector

                [reflectedX, reflectedY, reflectedZ] = ReflectRay( xDir, yDir, zDir, hitNormal); % get the reflected directions
                xDir = reflectedX;
                yDir = reflectedY;
                zDir = reflectedZ;

                impactCoord{iRay}(counter,:) = [currentX, currentY, currentZ];

                if size(impactCoord{iRay}, 1) == 1        
                    distance = sqrt((currentX -  DroneX)^2 + (currentY -  DroneY)^2 + (currentZ -  DroneZ)^2);
                else
                    distance = sqrt((currentX -  impactCoord{iRay}(end-1,1))^2 + (currentY -  impactCoord{iRay}(end-1,2))^2 + (currentZ -  impactCoord{iRay}(end-1,3))^2);
                end

                RayDistance{iRay}(counter) =  distance;

                attenuation = distance * absoptionCoeffcient;  


                if sum(RayDistance{iRay}) >= 1

                    inverseLaw = 20 * log10(sum(RayDistance{iRay}) * 2) + 11;


                else   
                    inverseLaw = 0;
                end

                distLoss{iRay}(counter) = attenuation;

                splValue = NoiseSource - sum(distLoss{iRay}) - inverseLaw;

                if splValue <=0                            % stop the tracing when spl reaches zero
                    break
                end

                SPL{iRay}(counter,1) = splValue;           % Calculate SPL at each point

                ray_distance = ray_distance + distance;   

                if round(currentZ) <= 2 && ReceiverCells(iRay) == 0  
                    ReceiverCells(iRay,1) = SPL{iRay}(counter,1);
                end

                continue;
            end
        end


        impactCoord{iRay}(counter,:) = [currentX, currentY, currentZ];   

        if size(impactCoord{iRay}, 1) == 1
            distance = sqrt((currentX -  DroneX)^2 + (currentY -  DroneY)^2 + (currentZ -  DroneZ)^2);
        else
            distance = sqrt((currentX -  impactCoord{iRay}(end-1,1))^2 + (currentY -  impactCoord{iRay}(end-1,2))^2 + (currentZ -  impactCoord{iRay}(end-1,3))^2);
        end


        RayDistance{iRay}(counter) =  distance;

        attenuation = distance * absoptionCoeffcient;   


        if sum(RayDistance{iRay}) >= 1

            inverseLaw = 20 * log10(sum(RayDistance{iRay}) * 2) + 11;

        else           
            inverseLaw = 0;
        end

        distLoss{iRay}(counter) = attenuation;

        splValue = NoiseSource - sum(distLoss{iRay}) - inverseLaw;

        if splValue <=0                                    % stop the tracing when spl reaches zero
            break
        end

        SPL{iRay}(counter,1) = splValue;                   % Calculate SPL at each point

        ray_distance = ray_distance + distance;

        if round(currentZ) <= 2 && ReceiverCells(iRay) == 0  
            ReceiverCells(iRay,1) = SPL{iRay}(counter,1);
        end

    end
end



nonZeroReceiverCells = ReceiverCells(ReceiverCells ~= 0);
if ~isempty(nonZeroReceiverCells)
    Noise = 10 * log10(sum(10 .^ (nonZeroReceiverCells ./ 10)));
else
    Noise = 0;
end

end



function [xDir_r, yDir_r, zDir_r] = ReflectRay(xDir, yDir, zDir, hitNormal)   % to reflect the ray when hitting obstacle

normal_Vector = hitNormal;                                 % Use the provided hitNormal (normal to ray hit face)
normalVector = normal_Vector / norm(normal_Vector);

incidentVector = [xDir, yDir, zDir];                       % Use the reflection formula: R = I - 2 * dot(I, N) * N for specular reflection only(neglecting diffuse reflection)
reflectionVector = incidentVector - 2 * dot(incidentVector, normalVector) * normalVector;

reflectionVector = reflectionVector / norm(reflectionVector); % Normalize the reflection vector

xDir_r = reflectionVector(1);
yDir_r = reflectionVector(2);                              % Reflected direction
zDir_r = reflectionVector(3);
end



function [hit, hitFace] = checkHitFace(x, y, z)                % to check which face of obstacle was hit by ray

epsilon = 0.1;                                             

hit = false;
hitFace = 0;

checkAxis = @(val, axis, dir) abs(val - round(val)) < epsilon && ((val > round(val)) == dir);

if checkAxis(x, 'x', true)                                 
    hit = true;
    hitFace = 1;
elseif checkAxis(x, 'x', false)                         
    hit = true;
    hitFace = 2;
elseif checkAxis(y, 'y', true)                           
    hit = true;
    hitFace = 3;
elseif checkAxis(y, 'y', false)                           
    hit = true;
    hitFace = 4;
elseif checkAxis(z, 'z', true)                             
    hit = true;
    hitFace = 5;
elseif checkAxis(z, 'z', false)                           
    hit = true;
    hitFace = 6;
end
end



